﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;
using SharpDX.Direct2D1;
using SharpDX;
using System.Globalization;
using System.Reflection;
using System.IO;
using System.Text.RegularExpressions;
using Svg;

namespace kr.co.daims.DisplayComponents.ChartGroup3.HeadingIndicatorView
{
    using Brush = SharpDX.Direct2D1.Brush;
    using RectangleF = SharpDX.RectangleF;
    using TextFormat = SharpDX.DirectWrite.TextFormat;
    using TextLayout = SharpDX.DirectWrite.TextLayout;
    public partial class HeadingIndicatorControl : UserControl
    {

        #region [ 전역 상수 ]
        /// <summary>
        /// 클래스내에서 문자열 포멧팅에 사용되는 전역 문자열 버퍼
        /// </summary>
        private StringBuilder strFormatter = null;

        /// <summary>
        /// 방위각 표시문자.
        /// </summary>
        private string[] Direction = new string[] { "N", "E", "S", "W" };

        /// <summary>
        /// 45도 간격으로 그려질 삼각형 모양의 표시기
        /// </summary>
        private Vector3[] vertices = new Vector3[] { new Vector3(-0.5f, 0.5f, 0.0f), new Vector3(0.5f, 0.5f, 0.0f), new Vector3(0.0f, -0.5f, 0.0f) };

        /// <summary>
        ///  비행체 모양 SVG Vector Point String
        /// </summary>
        public readonly string SVG_PLANE = @"M255.142,291.815l-1.192-10.374l-0.341,0.952h-6.217l-0.34-0.95l-1.192,10.372l-19.387-4.698v-4.101
	                                            l10.276-6.53l-23.395-5.56v-7.648l27.007-17.773v-11.48l1.438-4.102l2.622,2.848l2.643-17.47l3.437-9.801l3.481,9.983l2.599,17.288
	                                            l2.622-2.849l1.438,4.102v11.48l27.007,17.773v7.648l-23.396,5.56l10.276,6.53v4.101L255.142,291.815z";
        /// <summary>
        /// face 크기는 Casting 크기의 95%
        /// </summary>
        private float faceSizeRatio = 0.02f;
        #endregion

        #region [ 전역 변수 ]
        private SvgPath plane;
        private float planeScaleX = 1;
        private float planeScaleY = 1;
        private System.Drawing.PointF origin;
        private struct State
        {
            public float CurrentValue;
            public int Precision ;
            public Brush CastingBrush, FaceBrush, MarkerBrush, LineBrush, AircraftBrush;
            // ValueFont : 현재값 표시 폰트,  AzimuthFont :방위각 표시 폰트
            public TextFormat ValueFont, AzimuthFont;

            public bool IsShowValue;
            public bool IsValue360;

        };

        State mState;
        bool mIsOnBatchMode = false;

        WindowRenderTarget mRenderTarget;
        public SharpDX.DirectWrite.Factory mFontFactory = new SharpDX.DirectWrite.Factory(SharpDX.DirectWrite.FactoryType.Shared);


        public float CurrentValue
        {
            get { return mState.CurrentValue; }
            set
            {
                mState.CurrentValue = value;
                Redraw();
            }
        }

        #region [ CastringBrush ] 
        private Color castingColor;
        public Brush CastingBrush
        {
            get { return mState.CastingBrush; }
            set
            {
                mState.CastingBrush = value;
                Redraw();
            }
        }
        public Color CastingColor
        {
            get
            {
                return castingColor;
            }
            set
            {
                castingColor = value;
                if (mRenderTarget != null)
                {
                    CastingBrush = new SolidColorBrush(mRenderTarget, castingColor);
                }
            }
        }
        #endregion
        #region [ FaceBrush ] 
        private Color faceColor;
        public Brush FaceBrush
        {
            get { return mState.CastingBrush; }
            set
            {
                mState.FaceBrush = value;
                Redraw();
            }
        }
        public Color FaceColor
        {
            get
            {
                return faceColor;
            }
            set
            {
                faceColor = value;
                if (mRenderTarget != null)
                {
                    FaceBrush = new SolidColorBrush(mRenderTarget, faceColor);
                }
            }
        }
        #endregion

        #region [ MarkerBrush ] 
        private Color markerColor;
        public Brush MarkerBrush
        {
            get { return mState.MarkerBrush; }
            set
            {
                mState.MarkerBrush = value;
                Redraw();
            }
        }
        public Color MarkerColor
        {
            get
            {
                return markerColor;
            }
            set
            {
                markerColor = value;
                if (mRenderTarget != null)
                {
                    MarkerBrush = new SolidColorBrush(mRenderTarget, markerColor);
                }
            }
        }
        #endregion

        #region [ AircraftBrush ] 
        private Color aircraftColor;
        public Brush AircraftBrush
        {
            get { return mState.AircraftBrush; }
            set
            {
                mState.AircraftBrush = value;
                Redraw();
            }
        }
        public Color AircraftColor
        {
            get
            {
                return aircraftColor;
            }
            set
            {
                aircraftColor = value;
                if (mRenderTarget != null)
                {
                    AircraftBrush = new SolidColorBrush(mRenderTarget, aircraftColor);
                }
            }
        }
        #endregion
        #region [ LineBrush ] 
        private Color lineColor;
        public Brush LineBrush
        {
            get { return mState.LineBrush; }
            set
            {
                mState.LineBrush = value;
                Redraw();
            }
        }
        public Color LineColor
        {
            get
            {
                return lineColor;
            }
            set
            {
                castingColor = value;
                if (mRenderTarget != null)
                {
                    LineBrush = new SolidColorBrush(mRenderTarget, lineColor);
                }
            }
        }
        #endregion

        public int precision;
        public int Precision
        {
            get
            {
                return mState.Precision;
            }
            set
            {
                mState.Precision = value;
                Redraw();
            }
        }
        public new TextFormat ValueFont
        {
            get { return mState.ValueFont; }
            set
            {
                mState.ValueFont = value;
                Redraw();
            }
        }

        public new TextFormat AzimuthFont
        {
            get { return mState.AzimuthFont; }
            set
            {
                mState.AzimuthFont = value;
                Redraw();
            }
        }

        
        #endregion

        #region [ 이벤트 ]
        public delegate void BatchRun(HeadingIndicatorControl control);
        #endregion
        public HeadingIndicatorControl()
        {
            
            strFormatter = new StringBuilder(1000);
            InitializeComponent();
            SharpDX.Configuration.EnableObjectTracking = true;
            CastingColor = Color.Black;
            FaceColor = Color.White;
            lineColor = Color.Red;
            MarkerColor = Color.Green;
            AircraftColor = Color.Cyan;
            mState.Precision = 2;
            origin = new System.Drawing.PointF(0, 0);
            var svg = new Svg.SvgDocument();
            var converter = TypeDescriptor.GetConverter(typeof(Svg.Pathing.SvgPathSegmentList));

            plane = new SvgPath()
            {
                ID = "plane",
                Fill = new SvgColourServer(System.Drawing.Color.Black),
                PathData = (Svg.Pathing.SvgPathSegmentList)converter.ConvertFrom(SVG_PLANE),
            };
        }
        private void OnLoad(object sender, EventArgs e)
        {
            Factory fact = new Factory(FactoryType.MultiThreaded);

            var targetProperties = new RenderTargetProperties(new PixelFormat(SharpDX.DXGI.Format.B8G8R8A8_UNorm, AlphaMode.Premultiplied));
            var hwndProperties = new HwndRenderTargetProperties();
            hwndProperties.Hwnd = mRenderControl.Handle;
            hwndProperties.PixelSize = new SharpDX.Size2(mRenderControl.ClientSize.Width, mRenderControl.ClientSize.Height);
            hwndProperties.PresentOptions = PresentOptions.None;
            mRenderTarget = new WindowRenderTarget(fact, targetProperties, hwndProperties);

            mState.CastingBrush = new SolidColorBrush(mRenderTarget, CastingColor);
            mState.FaceBrush = new SolidColorBrush(mRenderTarget, FaceColor);
            mState.LineBrush = new SolidColorBrush(mRenderTarget, LineColor);
            mState.MarkerBrush = new SolidColorBrush(mRenderTarget, MarkerColor);
            mState.AircraftBrush = new SolidColorBrush(mRenderTarget, AircraftColor);
            mState.IsShowValue = true;
            SetText();
        }

        public void Batch(BatchRun run)
        {
            if (run == null)
                return;

            Exception exception = null;

            try
            {
                mIsOnBatchMode = true;
                run(this);
            }
            catch (Exception ex)
            {
                //exception = new Exception("BatchRun threw a exception", e);
                strFormatter.Clear();
                strFormatter.AppendFormat(CultureInfo.CreateSpecificCulture("en-US"), $"{ex.Message} :: {ex.StackTrace}");
#if (DEBUG)           
                CDebug.WriteError($"{MethodBase.GetCurrentMethod().DeclaringType.Name} :: {MethodBase.GetCurrentMethod().Name} :: {ex.Message} :: {ex.StackTrace}");
#endif
            }

            mIsOnBatchMode = false;
            Redraw(true);

            if (exception != null)
                throw exception;
        }

        private void Redraw(bool forced = false)
        {
            if (!mIsOnBatchMode || forced)
                mRenderControl.Invalidate();
        }
        
        private void Render(WindowRenderTarget target, int width, int height)
        {
            try
            {
                var diameter = Math.Min(width, height);
                
                target.Resize(new Size2(width, height));
                target.BeginDraw();
                //배경색을 희색으로 클리어
                target.Clear(SharpDX.Color.White);

                float widthRadius = width / 2.0f;
                float heightRadius = height / 2.0f;
                planeScaleX = width / 200.0f * 0.2f;
                planeScaleY = height / 200.0f * 0.2f;

                var centerVector = new Vector2(widthRadius, height / 2);
                var castingWidth = widthRadius * faceSizeRatio;
                // 눈금의 너비
                var graduationWidth = widthRadius * 0.005f;
                target.Transform = Matrix3x2.Identity;
                var t = target.Transform;
                target.Transform = t;
                using (var layer = new Layer(target))
                {
                    var layerParam = new LayerParameters()
                    {
                        ContentBounds = new RectangleF(0, 0, width, height)
                        ,
                        Opacity = 1.0f
                    };
                    
                    //Face Ellipse를 그림
                    target.FillEllipse(new Ellipse(centerVector, widthRadius, heightRadius), mState.FaceBrush);
                    #region [ Draw Aricraft ] 
                    target.DrawLine(new Vector2(widthRadius, (diameter / 2) * 0.1f), new Vector2(widthRadius-1, (diameter / 2) * 0.93f), mState.AircraftBrush);
                    var aircraft = new PathGeometry(target.Factory);

                    origin.X = 0;
                    origin.Y = 0;
                    using (var shink = aircraft.Open())
                    {
                        foreach (var p in plane.PathData.ToList())
                        {
                            if (p.GetType().Name == "SvgMoveToSegment")
                            {
                                origin = p.Start;
                                origin.X -=5;
                                origin.Y -= 50;
                                shink.BeginFigure(new Vector2((origin.X - p.Start.X) * planeScaleX, (p.Start.Y- origin.Y ) * planeScaleY), FigureBegin.Filled);
                            }
                            else if (p.GetType().Name == "SvgClosePathSegment")
                            {
                                shink.EndFigure(FigureEnd.Closed);
                            }
                            else
                            {
                                shink.AddLine(new Vector2((origin.X - p.End.X) * planeScaleX, (p.End.Y - origin.Y) * planeScaleY));
                            }
                        }
                        shink.Close();
                        target.Transform = Matrix3x2.Transformation(1, 1, 0, widthRadius, heightRadius);// 10 , 205.5
                        target.DrawGeometry(aircraft, mState.AircraftBrush);
                    }
                    #endregion
                    target.Transform = Matrix3x2.Rotation(MathUtil.DegreesToRadians(mState.CurrentValue), new Vector2(width / 2, height / 2));
                    target.PushLayer(ref layerParam, layer);
                    
                    //==================================
                    for (float i = 0; i < 360; i += 2.5f)
                    {
                        #region [ CurrValue 적용 파트 ] 
                        // Azimuth 각도에 따라 회전하는 기능이 필요한 부분
                        var radius = diameter / 2;
                        target.Transform = Matrix3x2.Rotation(MathUtil.DegreesToRadians(i + mState.CurrentValue), new Vector2(width / 2, height / 2));

                        target.DrawLine(new Vector2(width / 2, radius * 0.1f), new Vector2(width / 2, radius * (i % 5 == 0 ? 0.18f : 0.15f)), mState.LineBrush, (i % 5 == 0 ? graduationWidth * 2f : graduationWidth));
                        if (i % 30 == 0)
                        {
                            // 0,90,180,270도일때 각각 N, E, S, W 를 표시. 나머지 각에서는 30도 단위로 3,6,12,15,21,24,30,33을 표시
                            string valueString = i % 90 == 0 ? Direction[(int)(i / 90)].ToString() : ((int)(i / 30) * 3).ToString();

                            using (var azimuthText = new TextLayout(mFontFactory, valueString, mState.AzimuthFont, 300, 300))
                            {
                                target.DrawTextLayout(new Vector2((width - azimuthText.Metrics.Width) / 2, heightRadius * 0.2f), azimuthText, mState.LineBrush);
                            }
                        }
                        #endregion
                    }
                    target.PopLayer();
                }
                for (int i = 0; i < 360; i += 45)
                {
                    var triangle = new PathGeometry(target.Factory);
                    target.Transform = Matrix3x2.Rotation(MathUtil.DegreesToRadians(i), new Vector2(width / 2, height / 2));
                    using (var shink = triangle.Open())
                    {
                        var rectWith = width * 0.8f;
                        var rectHeight = height * 0.05f;
                        var rectBase = height * 0.99f;
                        var triSzie = rectWith * 0.03f;
                        shink.BeginFigure(new Vector2(width / 2 - triSzie / 2, rectBase), FigureBegin.Filled);
                        shink.AddLine(new Vector2(width / 2 + triSzie / 2, rectBase));
                        shink.AddLine(new Vector2(width / 2, (float)(height - rectHeight)));
                        shink.EndFigure(FigureEnd.Closed);
                        shink.Close();
                        target.FillGeometry(triangle, mState.MarkerBrush);
                    }
                }
                target.Transform = Matrix3x2.Identity;
                string precisionStr = String.Format($"F{mState.Precision}");
                if (mState.IsShowValue)
                {
                    float currValue = mState.IsValue360 ? mState.CurrentValue % 360 : mState.CurrentValue;
                    using (var currValueText = new TextLayout(mFontFactory, currValue.ToString(precisionStr), mState.AzimuthFont, 300, 300))
                    {
                        target.DrawTextLayout(new Vector2((width - currValueText.Metrics.Width) / 2, heightRadius * 1.2f), currValueText, mState.MarkerBrush);
                    }
                }

                target.DrawEllipse(new Ellipse(centerVector, (diameter / 2) *0.98f, (diameter / 2) * 0.98f), mState.CastingBrush, castingWidth);
                target.Flush();
                target.EndDraw();
                Console.WriteLine($"Draw Update {mState.CurrentValue}");
            }
            catch (Exception ex)
            {
                strFormatter.Clear();
                strFormatter.AppendFormat(CultureInfo.CreateSpecificCulture("en-US"), $"{ex.Message} :: {ex.StackTrace}");
#if (DEBUG)           
                CDebug.WriteError($"{MethodBase.GetCurrentMethod().DeclaringType.Name} :: {MethodBase.GetCurrentMethod().Name} :: {ex.Message} :: {ex.StackTrace}");
#endif
            }

        }
        

        private void OnResize(object sender, EventArgs e)
        {
            Redraw(true);
            SetText();
        }

        

        private void OnPaint(object sender, PaintEventArgs e)
        {
            Render(mRenderTarget, mRenderControl.ClientSize.Width, mRenderControl.ClientSize.Height);
        }

        #region [ Mouse Event ] 
        private void mRenderControl_MouseDown(object sender, MouseEventArgs e)
        {
            base.OnMouseDown(e);
        }

        private void mRenderControl_MouseUp(object sender, MouseEventArgs e)
        {
            base.OnMouseUp(e);
        }

        private void mRenderControl_MouseMove(object sender, MouseEventArgs e)
        {
            base.OnMouseMove(e);
        }

        private void mRenderControl_Resize(object sender, EventArgs e)
        {
            Redraw(true);

        }

        private void mRenderControl_DragEnter(object sender, DragEventArgs e)
        {
            base.OnDragEnter(e);
        }

        private void mRenderControl_DragDrop(object sender, DragEventArgs e)
        {
            base.OnDragDrop(e);
        }
        #endregion

        #region [ 사용자 함수]

        /// <summary>
        /// 방위각 Text 및 현재값 Text 를 업데이트 하기 위한 함수.
        /// 컴포넌트 크기에 따른  Text 동적 크기 설정.
        /// </summary>
        private void SetText()
        {
            var newSize = 10 + 2 * (this.Size.Width / 60);
            if (this.mState.AzimuthFont == null || this.mState.AzimuthFont.FontSize != newSize)
            {
                this.mState.AzimuthFont = new TextFormat(mFontFactory, "", newSize);
                //this.mState.ValueFont = new TextFormat(mFontFactory, "", newSize);
            }
        }

        #endregion
    }
}

